﻿#include "precompiled.h"
#include "common.h"
#include "Lens.h"

#include "Texture1D.h"

namespace RTCam {

// Default camera constraints
const float kInitialFocalLength = 50 * kMmToMeters;
const float kMinFocalLength = 10 * kMmToMeters;
const float kMaxFocalLength = 300 * kMmToMeters;

// Arbitrary limit on how far lens can be moved
// (Helps clamp the closest distance to which the lens can be focused)
const float kMaxLensDistanceFactor = 5;

// Epsilon for lens distance, as a factor of lens focal length
// (e.g. 50mm lens = 0.005mm epsilon)
const float kLensDistanceEpsilon = 0.0001f;

// Minimum distance that the camera can try to focus to
const float kMinFocalDistance = 50 * kMmToMeters;

Lens::Lens(void) :
	m_sphericalAberrationFactor(0),
	m_lensFocalLength(kInitialFocalLength),
	m_minFocalLength(kMinFocalLength),
	m_maxFocalLength(kMaxFocalLength),
	m_minFocalDistance(kMinFocalDistance),
	m_curLensDistance(kInitialFocalLength)
{
	m_curLensDistance = ValidateLensDistance(m_lensFocalLength);
}

Lens::~Lens(void)
{
}

float Lens::GetLensDistance()
{
	return m_curLensDistance;
}

void Lens::SetLensDistance(float lensDistance)
{
	m_curLensDistance = ValidateLensDistance(lensDistance);
}

float Lens::ValidateLensDistance(float lensDistance)
{
	float minLength = m_lensFocalLength * (1 + kLensDistanceEpsilon);
	float maxLength = m_lensFocalLength * kMaxLensDistanceFactor;

	return Clamp(lensDistance, minLength, maxLength);
}

float Lens::GetFocalDistance()
{
	return CalcFocalDistance(m_lensFocalLength, m_curLensDistance);
}

void Lens::SetFocalDistance( float focalDistance )
{
	float lensDistance = CalcLensDistance(m_lensFocalLength, focalDistance);
	SetLensDistance(lensDistance);
}

float Lens::ValidateFocalDistance( float focalDistance )
{
	// Make sure the focal distance doesn't get too close.
	focalDistance = std::max(focalDistance, m_minFocalDistance);
	
	// Clamp the focal distance to a valid value by converting to lens distance and validating that.
	float lensDistance = CalcLensDistance(m_lensFocalLength, focalDistance);
	lensDistance = ValidateLensDistance(lensDistance);
	focalDistance = CalcFocalDistance(m_lensFocalLength, lensDistance);

	return focalDistance;
}

float Lens::CalcFocalDistance( float focalLength, float lensDistance )
{
	return (focalLength * lensDistance) / (lensDistance - focalLength);
}

float Lens::CalcLensDistance( float focalLength, float focalDistance )
{
	return (focalLength * focalDistance) / (focalDistance - focalLength);
}

float Lens::GetFocalLength()
{
	return m_lensFocalLength;
}

void Lens::SetFocalLength(float focalLength)
{
	// Try to maintain the same focal distance
	float oldFocalDistance = GetFocalDistance();

	// Set the new focal length
	m_lensFocalLength = ValidateFocalLength(focalLength);
	
	// Try to restore the old focal distance
	SetFocalDistance(oldFocalDistance);
}

float Lens::ValidateFocalLength(float focalLength)
{
	return Clamp(focalLength, m_minFocalLength, m_maxFocalLength);
}

void Lens::SetFocalRanges(float minFocalLength, float maxFocalLength, float minFocalDistance)
{
	m_minFocalLength = minFocalLength;
	m_maxFocalLength = maxFocalLength;
	m_minFocalDistance = minFocalDistance;

	m_lensFocalLength = ValidateFocalLength(m_lensFocalLength);
	m_curLensDistance = ValidateLensDistance(m_curLensDistance);
}

} // end namespace